//package globalExamples;

import oracle.olapi.data.source.BooleanSource;
import oracle.olapi.data.source.DataProvider;
import oracle.olapi.data.source.FundamentalMetadataObject;
import oracle.olapi.data.source.FundamentalMetadataProvider;
import oracle.olapi.data.source.NumberSource;
import oracle.olapi.data.source.Source;
import oracle.olapi.data.source.StringSource;

import oracle.olapi.metadata.mdm.MdmAttribute;
import oracle.olapi.metadata.mdm.MdmHierarchy;
import oracle.olapi.metadata.mdm.MdmLevelHierarchy;
import oracle.olapi.metadata.mdm.MdmMeasure;
import oracle.olapi.metadata.mdm.MdmPrimaryDimension;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Complete code for Example 6-1 through Example 6-9 in Chapter 6, Understanding 
 * Source Objects chapter in the Oracle OLAP Developer's Guide to the OLAP API.
 *
 * This program uses the Context10g class, which uses the
 * CursorPrintWriter class.
 */
public class UnderstandingSourceObjects
{

private Context10g context = null;
private DataProvider dp = null;

  public UnderstandingSourceObjects()
  {
  }

  public void run(String [] args)
  {
    context = new Context10g(args, false);
    dp = context.getDataProvider();

    // Example 6-1
    Source prodDim = gettingTheDataTypeOfASource();

    // Example 6-2
    ArrayList sourcesList = usingTheisSubtypeOfMethod();

    // Informal examples
    gettingTheIDofASource(prodDim);
    Source prodSel = fullJoinExample(sourcesList);

    // Example 6-3
    Source custSel = producingASourceWithoutAnOutput();

    // Example 6-4
    producingASourceWithAnOutput(prodSel, custSel);

    // Example 6-5
    matchingSourcesToInputs();

    // Example 6-6
    usingShortcuts();

    // Example 6-7
    matchingTheBaseToInputOfJoined();

    // Example 6-8
    matchingInputOfBaseToOutputOfJoined();

    // Example 6-9
    matchingInputsOfAMeasureAndProducingOutputs();
  }

  private Source gettingTheDataTypeOfASource()
  {
    context.println("Example 6-1, Getting the Data Type of a Source\n");

    MdmPrimaryDimension mdmProdDim =
                          context.getMdmPrimaryDimensionByName("PRODUCT");

    FundamentalMetadataProvider fmp = dp.getFundamentalMetadataProvider();
    FundamentalMetadataObject fmoStringDataType = fmp.getStringDataType();
    Source stringDataTypeSource = fmoStringDataType.getSource();
    FundamentalMetadataObject fmoMdmProdDimDataType =
                                               mdmProdDim.getDataType();
    Source mdmProdDimDataTypeSource = fmoMdmProdDimDataType.getSource();
    Source prodDim = mdmProdDim.getSource();
    Source prodDimDataTypeSource = prodDim.getDataType();
    if(stringDataTypeSource == prodDimDataTypeSource &&
       mdmProdDimDataTypeSource == prodDimDataTypeSource)
    {
      System.out.println("The Source objects for the data types are all the " +
                     "same.");
    }
    else
    {  System.out.println("The Source objects for the data types are not " +
                         "all the same.");
    }

    return prodDim;
  }

  private ArrayList usingTheisSubtypeOfMethod()
  {
    context.println("Example 6-2, Using the isSubtypeOf Method\n");

    Source myList = dp.createListSource(new String[]
                                              {"PRODUCT_ROLLUP::FAMILY::4",
                                               "PRODUCT_ROLLUP::FAMILY::5",
                                               "PRODUCT_ROLLUP::FAMILY::7",
                                               "PRODUCT_ROLLUP::FAMILY::8"});


     MdmPrimaryDimension mdmProdDim =
                             context.getMdmPrimaryDimensionByName("PRODUCT");
     MdmHierarchy mdmProdRollup = mdmProdDim.getDefaultHierarchy();
     StringSource prodRollup = (StringSource) mdmProdRollup.getSource();

     // Selecting values using the selectValues shortcut.
     Source prodSel = prodRollup.selectValues(myList);
     if (prodSel.isSubtypeOf(prodRollup))
       context.println("prodSel is a subtype of prodRollup.");
     else
       context.println("prodSel is not a subtype of prodRollup.");

    context.println("The values of prodSel produced by the selectValues method");
    context.commit();
    context.displayResult(prodSel);

    ArrayList sourcesList = new ArrayList(2);
    sourcesList.add(prodRollup);
    sourcesList.add(myList);

    return sourcesList;
  }

  // Informal example between Example 6-2 and Example 6-3.
  // This method gets the ID of a Source.
  private void gettingTheIDofASource(Source prodDim)
  {
    context.println("\nInformal example that gets the ID of a Source.\n");
    System.out.println("The Source ID of prodDim is " + prodDim.getID());
  }

  // Informal example between Example 6-2 and Example 6-3.
  // This method uses the full join Method for the selectValues method.
  private Source fullJoinExample(ArrayList sourcesList)
  {
    context.println("\nUsing the full join Method for the selectValues.\n");

    Source prodRollup = (Source) sourcesList.get(0);
    Source myList = (Source) sourcesList.get(1);

    // Selecting values using the full join method, from the Inputs of
    // a Source section.
    Source prodSel = prodRollup.join(prodRollup.value(), myList,
                              Source.COMPARISON_RULE_SELECT, false);

   context.println("The values of prodSel produced by the full join method:");
   context.commit();
   context.displayResult(prodSel);

   return prodSel;
  }

  private Source producingASourceWithoutAnOutput()
  {
    context.println("\nExample 6-3, Using the join Method To Produce a Source " +
                    "Without an Output");

    MdmPrimaryDimension mdmProdDim =
                          context.getMdmPrimaryDimensionByName("CUSTOMER");
    MdmLevelHierarchy mdmShipRollup = (MdmLevelHierarchy)
                                       mdmProdDim.getDefaultHierarchy();

    Source shipRollup = mdmShipRollup.getSource();
    Source custValuesToSelect = dp.createListSource(new String[]
                                               {"SHIPMENTS_ROLLUP::REGION::9",
                                                "SHIPMENTS_ROLLUP::REGION::10"});
    Source shipRollupValues = shipRollup.value();
    Source custSel = shipRollup.join(shipRollupValues, custValuesToSelect,
                                 Source.COMPARISON_RULE_SELECT, false);

    context.println("The values of custSel:");
    context.commit();
    context.displayResult(custSel);

    return custSel;
  }

  private void producingASourceWithAnOutput(Source prodSel, Source custSel)
  {
    context.println("\nExample 6-4, Using the join Method To Produce a Source " +
                    "With an Output");

    Source custSelByProdSel = custSel.join(prodSel,
                                           dp.getEmptySource(),
                                           Source.COMPARISON_RULE_REMOVE,
                                           true);
    context.println("The values of custSelByProdSel:");
    context.commit();
    context.displayResult(custSelByProdSel);
  }

  private void matchingSourcesToInputs()
  {
    context.println("\nExample 6-5, Using the join Method To Match " +
                    "Source Objects To Input");

    MdmMeasure mdmUnitCost = context.getMdmMeasureByName("UNIT_COST");
    MdmPrimaryDimension mdmProdDim =
                             context.getMdmPrimaryDimensionByName("PRODUCT");
    MdmPrimaryDimension mdmTimeDim =
                             context.getMdmPrimaryDimensionByName("TIME");
    MdmLevelHierarchy mdmProdRollup = (MdmLevelHierarchy)
                                     mdmProdDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmCalendar = (MdmLevelHierarchy)
                                     mdmTimeDim.getDefaultHierarchy();

    Source unitCost = mdmUnitCost.getSource();
    Source calendar = mdmCalendar.getSource();
    Source prodRollup = mdmProdRollup.getSource();
    Source timeSel = calendar.join(calendar.value(),
                                    dp.createListSource(new String[]
                                    {"CALENDAR::MONTH::47",
                                     "CALENDAR::MONTH::59"}),
                                    Source.COMPARISON_RULE_SELECT,
                                    false);
    Source prodSel = prodRollup.join(prodRollup.value(),
                                     dp.createListSource(new String[]
                                     {"PRODUCT_ROLLUP::ITEM::13",
                                      "PRODUCT_ROLLUP::ITEM::14",
                                      "PRODUCT_ROLLUP::ITEM::15"}),
                                      Source.COMPARISON_RULE_SELECT,
                                      false);
    Source unitCostSel = unitCost.join(timeSel, dp.getEmptySource(),
                                       Source.COMPARISON_RULE_REMOVE,
                                       true)
                                 .join(prodSel, dp.getEmptySource(),
                                       Source.COMPARISON_RULE_REMOVE,
                                       true);

    context.println("The values of unitCostSel in Example 6-5:");
    context.commit();
    context.displayResult(unitCostSel);
  }

  private void usingShortcuts()
  {
    context.println("\nExample 6-6, Using Shortcuts");

    MdmMeasure mdmUnitCost = context.getMdmMeasureByName("UNIT_COST");
    MdmPrimaryDimension mdmProdDim =
                             context.getMdmPrimaryDimensionByName("PRODUCT");
    MdmPrimaryDimension mdmTimeDim =
                             context.getMdmPrimaryDimensionByName("TIME");
    MdmLevelHierarchy mdmProdRollup = (MdmLevelHierarchy)
                                     mdmProdDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmCalendar = (MdmLevelHierarchy)
                                     mdmTimeDim.getDefaultHierarchy();

    Source unitCost = mdmUnitCost.getSource();
    StringSource calendar = (StringSource) mdmCalendar.getSource();
    StringSource prodRollup = (StringSource) mdmProdRollup.getSource();
    Source timeSel = calendar.selectValues(new String[]
                                           {"CALENDAR::MONTH::47",
                                            "CALENDAR::MONTH::59"});
    Source prodSel = prodRollup.selectValues(new String[]
                                            {"PRODUCT_ROLLUP::ITEM::13",
                                             "PRODUCT_ROLLUP::ITEM::14",
                                             "PRODUCT_ROLLUP::ITEM::15"});
    Source unitCostSel = unitCost.join(timeSel)
                                 .join(prodSel);

    context.println("The values of unitCostSel in Example 6-6:");
    context.commit();
    context.displayResult(unitCostSel);
  }

  private void matchingTheBaseToInputOfJoined()
  {
    context.println("\nExample 6-7, Matching the Base Source to an Input " +
                    "of the Joined Source");

    MdmPrimaryDimension mdmProdDim =
                             context.getMdmPrimaryDimensionByName("PRODUCT");
     MdmHierarchy mdmProdRollup = mdmProdDim.getDefaultHierarchy();
     StringSource prodRollup = (StringSource) mdmProdRollup.getSource();

    Source myList = dp.createListSource(new String[]
                                        {"PRODUCT_ROLLUP::FAMILY::4",
                                         "PRODUCT_ROLLUP::FAMILY::5",
                                         "PRODUCT_ROLLUP::FAMILY::7",
                                         "PRODUCT_ROLLUP::FAMILY::8"});

    Source pos = dp.createListSource(new int[] {2, 4});
    Source myListPos = myList.position();
    Source myListSel = myList.join(myListPos, pos,
                                   Source.COMPARISON_RULE_SELECT, false);

    context.println("The values of myListSel in Example 6-7 with visible false:");
    context.commit();
    context.displayResult(myListSel);

    myListSel = myList.join(myListPos, pos,
                            Source.COMPARISON_RULE_SELECT, true);

    context.commit();
    context.println(" ");
    context.println("The values of myListSel in Example 6-7 with visible true:");
    context.displayResult(myListSel);
  }

  private void matchingInputOfBaseToOutputOfJoined()
  {
    context.println("\nExample 6-8, Matching an Input of the Base Source " +
                     "to an Output of the Joined Source");

    // Get the MdmMeasure for units
    MdmMeasure mdmUnits = context.getMdmMeasureByName("UNITS");
    // Get the Source for the measure
    Source units = mdmUnits.getSource();

    // Get the dimensions and their default hierarchies.
    MdmPrimaryDimension[] mdmPDims =
                             context.getMdmPrimaryDimensionsByName(new String[]
                                   {"PRODUCT", "CUSTOMER", "TIME", "CHANNEL"});

    MdmPrimaryDimension mdmProdDim = mdmPDims[0];
    MdmPrimaryDimension mdmCustDim = mdmPDims[1];
    MdmPrimaryDimension mdmTimeDim = mdmPDims[2];
    MdmPrimaryDimension mdmChanDim = mdmPDims[3];

    MdmLevelHierarchy mdmProdRollup = (MdmLevelHierarchy)
                                     mdmProdDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmShipRollup = (MdmLevelHierarchy)
                                     mdmCustDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmCalendar =(MdmLevelHierarchy)
                                     mdmTimeDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmChanRollup = (MdmLevelHierarchy)
                                     mdmChanDim.getDefaultHierarchy();

    // Get the short description attribute for the dimensions, and the
    // Source objects for the attributes.
    MdmAttribute mdmProdShortDescr =
                               mdmProdDim.getShortValueDescriptionAttribute();
    Source prodShortDescr = mdmProdShortDescr.getSource();
    MdmAttribute mdmCustShortDescr =
                               mdmCustDim.getShortValueDescriptionAttribute();
    Source custShortDescr = mdmCustShortDescr.getSource();
    MdmAttribute mdmTimeShortDescr =
                               mdmTimeDim.getShortValueDescriptionAttribute();
    Source timeShortDescr = mdmTimeShortDescr.getSource();
    // Get the short description attribute for the channel dimension.
    MdmAttribute mdmChanShortDescr =
                               mdmChanDim.getShortValueDescriptionAttribute();
    Source chanShortDescr = mdmChanShortDescr.getSource();

    // Get the Source objects for the default hierarchies of the dimensions
    // of the measures
    Source prodRollup = mdmProdRollup.getSource();
    Source custRollup = mdmShipRollup.getSource();
    Source calendar = mdmCalendar.getSource();
    Source chanRollup = mdmChanRollup.getSource();

    // Select values from the hierarchies
    Source prodSel = prodRollup.join(prodRollup.value(),
                                   dp.createListSource(new String[]
                                                 {"PRODUCT_ROLLUP::FAMILY::4",
                                                  "PRODUCT_ROLLUP::FAMILY::5"}),
                                   Source.COMPARISON_RULE_SELECT, false);
    Source custSel = custRollup.join(custRollup.value(),
                                   dp.createListSource(new String[]
                                              {"SHIPMENTS_ROLLUP::REGION::9",
                                               "SHIPMENTS_ROLLUP::REGION::10"}),
                                   Source.COMPARISON_RULE_SELECT, false);
    Source timeSel = calendar.join(calendar.value(),
                                   dp.createConstantSource("CALENDAR::YEAR::4"),
                                   Source.COMPARISON_RULE_SELECT, false);
    Source chanSel = chanRollup.join(chanRollup.value(),
                                   dp.createConstantSource(
                                               "CHANNEL_ROLLUP::CHANNEL::4"),
                                   Source.COMPARISON_RULE_SELECT, false);

    Source custSelByTime = custSel.join(timeSel, dp.getEmptySource(),
                                        Source.COMPARISON_RULE_REMOVE, true);

    Source prodByCustByTime = prodSel.join(custSelByTime, dp.getEmptySource(),
                                           Source.COMPARISON_RULE_REMOVE, true);

    Source selectedUnits = units.join(prodByCustByTime, dp.getEmptySource(),
                                        Source.COMPARISON_RULE_REMOVE, true)
                                .join(chanSel, dp.getEmptySource(),
                                        Source.COMPARISON_RULE_REMOVE, true);

    context.commit();
    context.displayResult(selectedUnits);

    // Create Source objects that have the short description attributes
    // as the type values and the dimension selection Source objects
    // as outputs
    Source prodSelDescr = prodShortDescr.join(prodSel);
    Source custSelDescr = custShortDescr.join(custSel);
    Source timeSelDescr = timeShortDescr.join(timeSel);
    Source chanSelDescr = chanShortDescr.join(chanSel);

    // Recreate the query including the short descriptions
    custSelByTime = custSelDescr.join(timeSelDescr,
                                      dp.getEmptySource(),
                                      Source.COMPARISON_RULE_REMOVE,
                                      true);

    prodByCustByTime = prodSelDescr.join(custSelByTime,
                                        dp.getEmptySource(),
                                        Source.COMPARISON_RULE_REMOVE,
                                        true);

    selectedUnits = units.join(prodByCustByTime, dp.getEmptySource(),
                               Source.COMPARISON_RULE_REMOVE, true)
                         .join(chanSelDescr, dp.getEmptySource(),
                               Source.COMPARISON_RULE_REMOVE, true);
    context.commit();
    context.displayResult(selectedUnits);
    //context.displayResultWithLocalValues(selectedUnits); // This works
  }

  private void matchingInputsOfAMeasureAndProducingOutputs()
  {
    context.println("\nExample 6-9, Matching the Inputs of a Measure " +
                    "and Producing Outputs");

    MdmMeasure mdmUnits = context.getMdmMeasureByName("UNITS");
    Source units = mdmUnits.getSource();

    MdmPrimaryDimension[] mdmPDims =
                             context.getMdmPrimaryDimensionsByName(new String[]
                                   {"PRODUCT", "CUSTOMER", "TIME", "CHANNEL"});

    MdmPrimaryDimension mdmProdDim = mdmPDims[0];
    MdmPrimaryDimension mdmCustDim = mdmPDims[1];
    MdmPrimaryDimension mdmTimeDim = mdmPDims[2];
    MdmPrimaryDimension mdmChanDim = mdmPDims[3];

    MdmLevelHierarchy mdmProdRollup = (MdmLevelHierarchy)
                                     mdmProdDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmShipRollup = (MdmLevelHierarchy)
                                     mdmCustDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmCalendar =(MdmLevelHierarchy)
                                     mdmTimeDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmChanRollup = (MdmLevelHierarchy)
                                     mdmChanDim.getDefaultHierarchy();


    Source prodRollup = mdmProdRollup.getSource();
    Source shipRollup = mdmShipRollup.getSource();
    Source calendar = mdmCalendar.getSource();
    Source chanRollup = mdmChanRollup.getSource();

    // create defaultHiers
    ArrayList defaultHiers = new ArrayList(4);
    defaultHiers.add(prodRollup);
    defaultHiers.add(shipRollup);
    defaultHiers.add(calendar);
    defaultHiers.add(chanRollup);

    Set inputs = units.getInputs();
    Iterator inputsItr = inputs.iterator();
    List outputs = units.getOutputs();
    Source input = null;

    int i = 1;
    System.out.println("The inputs of " + units.getID() + " are:");
    while(inputsItr.hasNext())
    {
      input = (Source) inputsItr.next();
      System.out.println(i + ": " + input.getID());
      i++;
    }

    System.out.println(" ");
    int setSize = inputs.size();
    for(i = 0; i < (setSize + 1); i++)
    {
      System.out.println(units.getID() + " has " + inputs.size() +
                       " inputs and " + outputs.size() + " outputs.");
      if (i < setSize)
      {
        input = (Source) defaultHiers.get(i);
        System.out.println("Joining " + input.getID() + " to "
                                      + units.getID());
        units = units.join(input);
        inputs = units.getInputs();
        outputs = units.getOutputs();
      }
    }

    System.out.println(" ");
    System.out.println("The outputs of " + units.getID() + " are:");
    Iterator outputsItr = outputs.iterator();
    i = 1;
    while(outputsItr.hasNext())
    {
      Source output = (Source) outputsItr.next();
      System.out.println(i + ": " + output.getID());
      i++;
    }
  }

  public static void main(String[] args)
  {
    new UnderstandingSourceObjects().run(args);
  }
}
